/* 
cutdown2litter.c
calculation of daily  mortality fluxes of cut-down biomass: these fluxes all enter litter sinks

*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
Biome-BGCMuSo v7.0.
Copyright 2025, D. Hidy [dori.hidy@gmail.com]
Hungarian Academy of Sciences, Hungary
See the website of Biome-BGCMuSo at http://nimbus.elte.hu/bbgc/ for documentation, model executable and example input files.
*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "ini.h"
#include "bgc_struct.h"
#include "bgc_func.h"
#include "bgc_constants.h"

int cutdown2litter(const soilprop_struct* sprop, const epconst_struct* epc, const epvar_struct* epv, cstate_struct* cs, cflux_struct* cf,nstate_struct* ns, nflux_struct* nf)
{
	int errorCode=0;
	int layer;
	double mort_nW, mort_W;

	mort_nW = epc->mort_CnW_to_litter;
	mort_W  = epc->mort_CW_to_litter;

	/**************************************************************/
	/* 1. mortality fluxes to litter pools */

	cf->CTDBc_leaf_to_litr     = cs->CTDBc_leaf     * mort_nW;
	cf->CTDBc_froot_to_litr    = cs->CTDBc_froot    * mort_nW;
	cf->CTDBc_yield_to_litr    = cs->CTDBc_yield    * mort_nW;
	cf->CTDBc_softstem_to_litr = cs->CTDBc_softstem * mort_nW;
	cf->CTDBc_cstem_to_cwd     = cs->CTDBc_cstem    * mort_W;
	cf->CTDBc_croot_to_cwd     = cs->CTDBc_croot    * mort_W;

    nf->CTDBn_leaf_to_litr     = ns->CTDBn_leaf     * mort_nW;
	nf->CTDBn_froot_to_litr    = ns->CTDBn_froot    * mort_nW;
	nf->CTDBn_yield_to_litr    = ns->CTDBn_yield    * mort_nW;
	nf->CTDBn_softstem_to_litr = ns->CTDBn_softstem * mort_nW;
	nf->CTDBn_cstem_to_cwd     = ns->CTDBn_cstem    * mort_W;
	nf->CTDBn_croot_to_cwd     = ns->CTDBn_croot    * mort_W;


	
		
	/* returning of cut-down plant material into litter */
	cf->CTDBc_to_litr  = cf->CTDBc_leaf_to_litr     + cf->CTDBc_froot_to_litr + cf->CTDBc_yield_to_litr + cf->CTDBc_softstem_to_litr +
						 cf->CTDBc_cstem_to_cwd     + cf->CTDBc_croot_to_cwd;

	nf->CTDBn_to_litr  = nf->CTDBn_leaf_to_litr     + nf->CTDBn_froot_to_litr + nf->CTDBn_yield_to_litr + nf->CTDBn_softstem_to_litr +
						 nf->CTDBn_cstem_to_cwd     + nf->CTDBn_croot_to_cwd;



	/****************************************************************************************/
	/* 2. mortality fluxes turn into litter pools: 	aboveground biomass into the top soil layer, belowground biomass divided between soil layers based on their root content */
	
	/* 2.1 aboveground biomass into the top soil layer */

	/* litter turns into the first three soil layers  (non-woody biomass: proportion to soil layer thickness, woody-biomass: higher propotion in layer2 */

	for (layer =0; layer < N_SOILLAYERS; layer++)
	{
		cs->litr1c[layer] += (cf->CTDBc_leaf_to_litr * epc->leaflitr_flab  + cf->CTDBc_yield_to_litr * epc->yieldlitr_flab  + 
						  cf->CTDBc_softstem_to_litr * epc->softstemlitr_flab) * sprop->PROPlayerDC[layer];
		cs->litr2c[layer] += (cf->CTDBc_leaf_to_litr * epc->leaflitr_fucel + cf->CTDBc_yield_to_litr * epc->yieldlitr_fucel + 
						  cf->CTDBc_softstem_to_litr * epc->softstemlitr_fucel) * sprop->PROPlayerDC[layer];
		cs->litr3c[layer] += (cf->CTDBc_leaf_to_litr * epc->leaflitr_fscel + cf->CTDBc_yield_to_litr * epc->yieldlitr_fscel + 
						  cf->CTDBc_softstem_to_litr * epc->softstemlitr_fscel) * sprop->PROPlayerDC[layer];
		cs->litr4c[layer] += (cf->CTDBc_leaf_to_litr * epc->leaflitr_flig  + cf->CTDBc_yield_to_litr * epc->yieldlitr_flig  + 
						  cf->CTDBc_softstem_to_litr * epc->softstemlitr_flig) * sprop->PROPlayerDC[layer];
		cs->cwdc[layer]   += (cf->CTDBc_cstem_to_cwd) * sprop->PROPlayerDC[layer];

		/* update of aboveground biomass */
		cs->litrCabove[layer] += (cf->CTDBc_leaf_to_litr + cf->CTDBc_yield_to_litr + cf->CTDBc_softstem_to_litr) * sprop->PROPlayerDC[layer];
		cs->cwdCabove[layer] += (cf->CTDBc_cstem_to_cwd) * sprop->PROPlayerDC[layer];

		ns->litr1n[layer] += (nf->CTDBn_leaf_to_litr * epc->leaflitr_flab  + nf->CTDBn_yield_to_litr * epc->yieldlitr_flab  + 
						  nf->CTDBn_softstem_to_litr * epc->softstemlitr_flab) * sprop->PROPlayerDC[layer];
		ns->litr2n[layer] += (nf->CTDBn_leaf_to_litr * epc->leaflitr_fucel + nf->CTDBn_yield_to_litr * epc->yieldlitr_fucel + 
						  nf->CTDBn_softstem_to_litr * epc->softstemlitr_fucel) * sprop->PROPlayerDC[layer];
		ns->litr3n[layer] += (nf->CTDBn_leaf_to_litr * epc->leaflitr_fscel + nf->CTDBn_yield_to_litr * epc->yieldlitr_fscel + 
						  nf->CTDBn_softstem_to_litr * epc->softstemlitr_fscel) * sprop->PROPlayerDC[layer];
		ns->litr4n[layer] += (nf->CTDBn_leaf_to_litr * epc->leaflitr_flig  + nf->CTDBn_yield_to_litr * epc->yieldlitr_flig  + 
						  nf->CTDBn_softstem_to_litr * epc->softstemlitr_flig) * sprop->PROPlayerDC[layer];
		ns->cwdn[layer]   += (nf->CTDBn_cstem_to_cwd) * sprop->PROPlayerDC[layer];
	}


	/* 2.2 	belowground biomass divided between soil layers based on their root content */ 
	if (epv->rootDepth > CRIT_PREC)
	{
		for (layer=0; layer < N_SOILLAYERS; layer++)
		{
			cs->litr1c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_flab  * epv->rootlengthProp[layer];
			cs->litr2c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_fucel * epv->rootlengthProp[layer];
			cs->litr3c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_fscel * epv->rootlengthProp[layer];
			cs->litr4c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_flig  * epv->rootlengthProp[layer];
			cs->cwdc[layer]    += cf->CTDBc_croot_to_cwd  * epv->rootlengthLandD_prop[layer];

			ns->litr1n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_flab  * epv->rootlengthProp[layer];
			ns->litr2n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_fucel * epv->rootlengthProp[layer];
			ns->litr3n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_fscel * epv->rootlengthProp[layer];
			ns->litr4n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_flig  * epv->rootlengthProp[layer];
			ns->cwdn[layer]    += nf->CTDBn_croot_to_cwd  * epv->rootlengthLandD_prop[layer];
		}
	}
	else
	{
		for (layer=0; layer < N_SOILLAYERS; layer++)
		{
			cs->litr1c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_flab  * epv->rootlengthLandD_prop[layer];
			cs->litr2c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_fucel * epv->rootlengthLandD_prop[layer];
			cs->litr3c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_fscel * epv->rootlengthLandD_prop[layer];
			cs->litr4c[layer]  += cf->CTDBc_froot_to_litr * epc->frootlitr_flig  * epv->rootlengthLandD_prop[layer];
			cs->cwdc[layer]    += cf->CTDBc_croot_to_cwd  * epv->rootlengthLandD_prop[layer];

			ns->litr1n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_flab  * epv->rootlengthLandD_prop[layer];
			ns->litr2n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_fucel * epv->rootlengthLandD_prop[layer];
			ns->litr3n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_fscel * epv->rootlengthLandD_prop[layer];
			ns->litr4n[layer]  += nf->CTDBn_froot_to_litr * epc->frootlitr_flig  * epv->rootlengthLandD_prop[layer];
			ns->cwdn[layer]    += nf->CTDBn_croot_to_cwd  * epv->rootlengthLandD_prop[layer];
		}
	}


	
	/************************************************************/
	/* 3. decreasing of temporary pool */

	cs->CTDBc_leaf     -= cf->CTDBc_leaf_to_litr;
	cs->CTDBc_froot    -= cf->CTDBc_froot_to_litr;
	cs->CTDBc_yield    -= cf->CTDBc_yield_to_litr;
	cs->CTDBc_softstem -= cf->CTDBc_softstem_to_litr;
	cs->CTDBc_cstem    -= cf->CTDBc_cstem_to_cwd;
	cs->CTDBc_croot    -= cf->CTDBc_croot_to_cwd;
	

    ns->CTDBn_leaf     -= nf->CTDBn_leaf_to_litr;
	ns->CTDBn_froot    -= nf->CTDBn_froot_to_litr;
	ns->CTDBn_yield    -= nf->CTDBn_yield_to_litr;
	ns->CTDBn_softstem -= nf->CTDBn_softstem_to_litr;
	ns->CTDBn_cstem    -= nf->CTDBn_cstem_to_cwd;
	ns->CTDBn_croot    -= nf->CTDBn_croot_to_cwd;


	/************************************************************/
	/* 4. precision control */

	if ((cs->CTDBc_leaf != 0 && fabs(cs->CTDBc_leaf) < CRIT_PREC) || (ns->CTDBn_leaf != 0 && fabs(ns->CTDBn_leaf) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_leaf;
		ns->FIREsnk_N += ns->CTDBn_leaf; 
		cs->CTDBc_leaf = 0;
		ns->CTDBn_leaf = 0;
	}

	if ((cs->CTDBc_yield != 0 && fabs(cs->CTDBc_yield) < CRIT_PREC) || (ns->CTDBn_yield != 0 && fabs(ns->CTDBn_yield) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_yield;
		ns->FIREsnk_N += ns->CTDBn_yield; 
		cs->CTDBc_yield = 0;
		ns->CTDBn_yield = 0;
	}

	if ((cs->CTDBc_softstem != 0 && fabs(cs->CTDBc_softstem) < CRIT_PREC) || (ns->CTDBn_softstem != 0 && fabs(ns->CTDBn_softstem) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_softstem;
		ns->FIREsnk_N += ns->CTDBn_softstem; 
		cs->CTDBc_softstem = 0;
		ns->CTDBn_softstem = 0;
	}

	if ((cs->CTDBc_froot != 0 && fabs(cs->CTDBc_froot) < CRIT_PREC) || (ns->CTDBn_froot != 0 && fabs(ns->CTDBn_froot) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_froot;
		ns->FIREsnk_N += ns->CTDBn_froot; 
		cs->CTDBc_froot = 0;
		ns->CTDBn_froot = 0;
	}

	

	if ((cs->CTDBc_cstem != 0 && fabs(cs->CTDBc_cstem) < CRIT_PREC) || (ns->CTDBn_cstem != 0 && fabs(ns->CTDBn_cstem) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_cstem;
		ns->FIREsnk_N += ns->CTDBn_cstem; 
		cs->CTDBc_cstem = 0;
		ns->CTDBn_cstem = 0;
	}

	if ((cs->CTDBc_croot != 0 && fabs(cs->CTDBc_croot) < CRIT_PREC) || (ns->CTDBn_croot != 0 && fabs(ns->CTDBn_croot) < CRIT_PREC))
	{
		cs->FIREsnk_C += cs->CTDBc_croot;
		ns->FIREsnk_N += ns->CTDBn_croot; 
		cs->CTDBc_croot = 0;
		ns->CTDBn_croot = 0;
	}


	return(errorCode);
}